/*
  (c) 2005-2009 Philipp Klaus Krause philipp@colecovision.eu

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.
   
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
   
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

*/

#include <iostream>
#include <fstream>
#include <ios>

#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

// Modify notes by halftones.
signed int global_modifiers[7] = {0, 0, 0, 0, 0, 0, 0};
signed int local_modifiers[4][7] = {{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}};
bool local_modifier_enabled[4][7] = {{false, false, false, false, false, false, false}, {false, false, false, false, false, false, false}, {false, false, false, false, false, false, false}, {false, false, false, false, false, false, false}};
const signed int halftones[7] = {0, 2, 4, 5, 7, 9, 11};
const unsigned char C = 0;
const unsigned char D = 1;
const unsigned char E = 2;
const unsigned char F = 3;
const unsigned char G = 4;
const unsigned char A = 5;
const unsigned char B = 6;
const unsigned char Z = 23;

const unsigned int LENGTH_SCALE = 16;

std::ofstream ofile;

unsigned int default_note;

unsigned int length;
unsigned int tlength;

unsigned int legato;

unsigned long oldlinecounter, linecounter;
bool skip;

bool verbose = false;
bool exclamationlinebreak = false; // Enable old ! linebreak syntax
bool plusdecoration = false; // Enable old + decoration syntax

static struct
{
	unsigned char base;
	signed int oktave;
	signed int local_modifier;
	bool local_modifier_enabled;
	char num[16];
	bool denom_current;
	unsigned int counter;
	char denom[16];
	unsigned char loudness;
	bool accent;
	bool local_legato;
	bool staccato;
	bool dotted;
	bool halved;
	bool grace;
	bool graced;
	unsigned int grace_length;
	bool acciaccatura;
} note;

void erase(void)
{
	note.base = 0;
	note.local_modifier = 0;
	note.local_modifier_enabled = false;
	note.denom_current = false;
	memset(note.num, 0, 16);
	memset(note.denom, 0, 16);
	note.counter = 0;
	note.accent = false;
	note.local_legato = false;
	note.staccato = false;
	note.dotted = false;
	note.halved = false;
	note.graced = false;
	note.acciaccatura = false;
}

void line(void)
{
	std::cout << "Line " << linecounter << ": ";
}

void unimplemented_key(void)
{
	line();
	std::cerr << "Unimplemented key.\n";
	std::cerr << "Supported keys:\nC, G, D, A, E, B, F#, C#, F, Bb, Eb, Ab, Db, Gb, Cb (major scale),\nAm, Dm, Gm, Cm, Fm, Bm, and Em (minor scale),\nG mix, D mix, A mix, E mix, B mix, C mix, F mix (mixolydian scale), B dor, E dor, A dor, D dor, F dor, C dor, G dor (dorian scale).";
}

void translate(void)
{
	if(!note.base) return;
	if(skip)
	{
		erase();
		return;
	}
	if(note.grace && note.graced)
	{
		line();
		std::cerr << "Warning: Ignoring unimplemented subsequent grace note sequence.\n";
		erase();
		note.graced = true;
		return;
	}
	length++;
	signed int transformed;
	switch(note.base)
	{
	case 'C':
		note.base = C;
		break;
	case 'D':
		note.base = D;
		break;
	case 'E':
		note.base = E;
		break;
	case 'F':
		note.base = F;
		break;
	case 'G':
		note.base = G;
		break;
	case 'A':
		note.base = A;
		break;
	case 'B':
		note.base = B;
		break;
	case 'Z':
	case 'X':
		note.base = Z;
		break;
	default:
		erase();
		return;
	}
	signed char modifier;
	if(note.base != Z)
	{
		if(note.local_modifier_enabled)
		{
			modifier = note.local_modifier;
			local_modifiers[note.oktave][note.base] = modifier;
			local_modifier_enabled[note.oktave][note.base] = true;
		}
		else if(local_modifier_enabled[note.oktave][note.base])
		{
			modifier = local_modifiers[note.oktave][note.base];
		}
		else
			modifier = global_modifiers[note.base];
		transformed = halftones[note.base];
		transformed += modifier;
		if(transformed < 0)
		{
			transformed += 12;
			note.oktave--;
		}
		if(transformed >= 12)
		{
			transformed -= 12;
			note.oktave++;
		}
	}
	else	// Pause
	{
		note.oktave = 0;
		transformed = 0xf;
	}

	if(note.oktave < 0 || note.oktave > 7)
    {
    	std::cerr << "Warning: Pitch out of representable range.\n";
		note.oktave = 0;
		transformed = 0xf;
	}

	unsigned int length = default_note;
	unsigned int rel_length;
	if(note.num[0])
		length *= atoi(note.num);
	if(note.dotted)
		length *= 3;
	if(note.dotted || note.halved)
		length /= 2;
	if(note.denom_current && note.counter)
	{
		if(atoi(note.denom))
			length /= atoi(note.denom);
		else
		{
			line();
			std::cerr << "Warning: Invalid note length.\n";
		}
	}
	else if(note.denom_current)
		length /= note.denom[0];

	uint8_t tone_byte;
	if(note.acciaccatura)
{line(); std::cout << "acciaccatura\n";
		length = 0;
}
	if(!length)
	{
		if(note.grace)
			length = 1;
		else
		{
			line();
			std::cerr << "Warning: Note of length zero.\n";
		}
	}
	if(note.local_legato || legato)
		rel_length = 0;	// Full length.
	else if(note.staccato)
		rel_length = 1;	// 25%.
	else
		rel_length = 3;	// default length 75%.
	if(note.graced)
    {
		if(note.grace_length > length)
		{
			line();
			std::cerr << "Error grace note longer than note.\n";
		}
		else if(note.grace_length == length)
		{
			line();
			std::cerr << "Warning: Grace note of full lenth of note.\n";
		}
		length-= note.grace_length;
    }
	unsigned int loudness = note.loudness;
	if(note.accent && loudness < 3)
		loudness++;
	else if(note.accent)
		std::cerr << "Warning: Ignored accent within forte.";
	uint_fast8_t length_byte = ((length << 4) | (rel_length << 2) | loudness) & 0xff;
	uint16_t entry;
	tone_byte = (note.oktave << 4) | transformed;
	entry = (uint16_t(tone_byte) << 8) | uint16_t(length_byte);
	if(length)
		ofile << " " << entry << ",";
	tlength += length;
	erase();
	note.graced = note.grace;
	note.grace_length = length;
}

void usage(void)
{
	std::cout << "abc2cvm " << VERSION << "\n";
	std::cout << "Usage: abc2cvm [--verbose | -v][--voice | -V <voice>][--ref | -X <ref>] filename\n";
}

void key(std::istream& ifile, char end)
{
	char a = ' ', b = ' ', c = ' ', d = ' ';

	// Skip whitespace
	while(a == ' ' || a == '\t')
		a = ifile.get();

	while(b == ' ' || b == '\t')
		b = ifile.get();
	if(b != '\n')
		while(c == ' ' || c == '\t')
			c = ifile.get();
	if(b != '\n' && c != '\n')
		while(d == ' ' || d == '\t')
			d = ifile.get();

	if(b != '\n' && b != ' ' && b != '%' && b != 'b' && b != '#' && b != 'm' && b != 'M' && b != 'd' && b != 'D' && !(a == 'c' && b == 'l' && c == 'e' && d == 'f') && !(b == 'c' && c == 'l' && d == 'e'))
	{
		unimplemented_key();
		return;
	}

	if(a == 'c' && b == 'l' && c == 'e' && d == 'f')	// Clef
	{
		if(verbose)
		{
			line();
			std::cout << "Ignoring clef.\n";
		}
	}
	else if(b != 'b' && b != '#' && (b != 'm' && b !='M' || c == 'a' || c == 'A') && b != 'd' && b != 'D')	// Major scales B, E, A, D, G, C, F.
	{
		if(verbose)
			std::cout << "Assuming major scale.\n";
		switch(a)
		{
		case 'B':
			global_modifiers[A] = 1;
		case 'E':
			global_modifiers[D] = 1;
		case 'A':
			global_modifiers[G] = 1;
		case 'D':
			global_modifiers[C] = 1;
		case 'G':
			global_modifiers[F] = 1;
		case 'C':
			break;
		case 'F':
			global_modifiers[B] = -1;
			break;
		default:
			unimplemented_key();
			return;
		}
	}
	else if(b == 'b' && (c != 'm' && c != 'M' || d == 'a' || d == 'A') && b != 'd' && b != 'D')	// Major scales Cb, Gb, Db, Ab, Eb, Bb.
	{
		if(verbose)
			std::cout << "Assuming major scale.\n";
		switch(a)
		{
		case 'C':
			global_modifiers[A] = -1;
		case 'G':
			global_modifiers[C] = -1;
		case 'D':
			global_modifiers[G] = -1;
		case 'A':
			global_modifiers[D] = -1;
		case 'E':
			global_modifiers[A] = -1;
		case 'B':
			global_modifiers[E] = -1;
			global_modifiers[B] = -1;
			break;
		default:
			unimplemented_key();
			return;
		}
	}
	else if((b == 'm' || b == 'M') && (d != 'x' && d != 'X'))	// Minor scales Fm, Cm, Gm, Dm, Am, Bm, Em.
	{
		if(verbose)
			std::cout << "Assuming minor scale.\n";
		switch(a)
		{
		case 'F':
			global_modifiers[D] = -1;
		case 'C':
			global_modifiers[A] = -1;
		case 'G':

			global_modifiers[E] = -1;
		case 'D':
			global_modifiers[B] = -1;
		case 'A':
			break;
		case 'B':
			global_modifiers[F] = 1;
		case 'E':
			global_modifiers[C] = 1;
			break;
		default:
			unimplemented_key();
			return;
		}
	}
	else if((b == 'm' || b == 'M') && (c == 'i' || c == 'I') && (d == 'x' || c == 'X'))	// Mixolydian G Mix, D Mix, A Mix, E Mix, B Mix, C Mix, F Mix,.
	{
		if(verbose)
			std::cout << "Assuming mixolydian scale.\n";
		switch(a)
		{
		case 'B':
			global_modifiers[D] = 1;
		case 'E':
			global_modifiers[G] = 1;
		case 'A':
			global_modifiers[C] = 1;
		case 'D':
			global_modifiers[F] = 1;
		case 'G':
			break;
		case 'F':
			global_modifiers[E] = -1;
		case 'C':
			global_modifiers[B] = -1;
			break;
		default:
			unimplemented_key();
			return;
		}
	}
	else if(b == '#' && (c != 'm' && c != 'M' || d == 'a' || d == 'A'))	// Major scales C#, F#.
	{
		if(verbose)
			std::cout << "Assuming major scale.\n";
		switch(a)
		{
		case 'C':
			global_modifiers[B] = 1;
		case 'F':
			global_modifiers[E] = 1;
			global_modifiers[A] = 1;
			global_modifiers[D] = 1;
			global_modifiers[G] = 1;
			global_modifiers[C] = 1;
			global_modifiers[F] = 1;
			break;
		default:
			unimplemented_key();
			return;
		}
	}
	else if(b == '#' && (c == 'm' || c != 'M'))	// Minor scales F#, C#, G#, D#, A#.
	{
		if(verbose)
			std::cout << "Assuming minor scale.\n";
		switch(a)
		{
		case 'A':
			global_modifiers[B] = 1;
		case 'D':
			global_modifiers[E] = 1;
		case 'G':
			global_modifiers[A] = 1;
		case 'C':
			global_modifiers[D] = 1;
		case 'F':
			global_modifiers[G] = 1;
			global_modifiers[C] = 1;
			global_modifiers[F] = 1;
			break;
		default:
			unimplemented_key();
			return;
		}
	}
	else if(b == 'b' && (c == 'm' || c == 'M'))	// Minor scales Bbm, Ebm, Abm.
	{
		if(verbose)
			std::cout << "Assuming minor scale.\n";
		switch(a)
		{
		case 'A':
			global_modifiers[A] = -1;
		case 'E':
			global_modifiers[C] = -1;
		case 'B':
			global_modifiers[G] = -1;
			global_modifiers[D] = -1;
			global_modifiers[A] = -1;
			global_modifiers[E] = -1;
			global_modifiers[B] = -1;
			break;
		default:
			unimplemented_key();
			return;
		}
	}
	else if(b == 'd' || b == 'D')
	{
		if(verbose)
			std::cout << "Assuming dorian scale.\n";

		switch(a)
		{
		case 'B':
			global_modifiers[G] = 1;
		case 'E':
			global_modifiers[C] = 1;
		case 'A':
			global_modifiers[F] = 1;
		case 'D':
			break;
		case 'F':
			global_modifiers[A] = -1;
			break;
		case 'C':
			global_modifiers[E] = -1;
			break;
		case 'G':
			global_modifiers[B] = -1;
			break;
		default:
			unimplemented_key();
			return;
		}
	}
	else
	{
		unimplemented_key();
		return;
	}

	if(b == '\n' || c == '\n' || d == '\n')
		linecounter++;

	if(c == end || d == end)
		b = end;

	// Ignore clef.
	while(b != end && b != EOF)
	{
		b = ifile.get();
		if(b == '\n')
			linecounter++;
	}
}

int main(int argc, char **argv)
{
	std::string ifilename, ofilename, voice, x;
	std::streampos repeatmark;
	bool repeatflag;

	legato = 0;
	length = 0;
	tlength = 0;
	skip = false;
	default_note = 0;

	int i;
	if(argc == 1)
	{
		std::cerr << "Error: Not enough arguments.\n";
		usage();
		return(-1);
	}
	for(i = 1; i < argc - 1; i++)
	{
		std::string stra = argv[i];
		if(stra == "--verbose" || stra == "-v")
		{
			verbose = true;
		}
		else if((stra == "--voice" || stra == "-V") && ++i < argc - 1)
		{
			voice = argv[i];
		}
		else if((stra == "--ref" || stra == "-X") && ++i < argc - 1)
		{
			x = argv[i];
		}
		else
		{
			usage();
			return(-1);
		}
	}

	ifilename = argv[argc - 1];
	if(verbose) std::cout << "Using " << ifilename << " as input file.\n";
	if(voice == "")
		ofilename = ifilename + ".music.c";
	else if (x == "")
		ofilename = ifilename + "." + voice + ".c";
	else
		ofilename = ifilename + "." + x + "." + voice + ".c";
	if(verbose) std::cout << "Using " << ofilename << " as output file.\n";

	std::ifstream ifile; 
	ifile.open(ifilename.c_str(), std::ios::in);
	if(!ifile.is_open())
	{
		std::cerr << "Error: Failed to open input file " << ifilename << ".\n";
		return(-1);
	}
	ofile.open(ofilename.c_str(), std::ios::out | std::ios::trunc);

	linecounter = 1;
	char a;

	if(x != "")
	{
		while((a = ifile.get()) != EOF)
		{
			if(a == '\n')
				linecounter++;
			if(a != 'X')
				continue;
			ifile.get();	// Get ':'
			
			std::string x2;
			do
				a = ifile.get();
			while(a == ' ');

			do
			{
				x2 += a;
				a = ifile.get();
			}
			while(a != ' ' && a != '\n');	
			if(a == '\n')
				linecounter++;

			if(x == x2)
				break;
		}
		if(a == EOF)
		{
			std::cerr << "Error: Specified tune " << x << " not found.\n";
			return(-1);
		}
	}

	while((a = ifile.get()) != EOF)
	{
		if(a == 'K')
		{
			ifile.get();	// Get ':'
			key(ifile, '\n');
			break;
		}
		else if(a == 'M')	// Set default note length
		{
			if(default_note)
			{
				while(ifile.get() != '\n');
				linecounter++;
				continue;
			}

			if(ifile.get() != ':') continue;
			char temp[16] = "               ";
			unsigned int ltmp;
			i = 0;
			while((a = ifile.get()) != '/')
			{
				if(a == ' ') continue;
				if(a < '0' || a > '9')
					break;
				temp[i] = a;
				i++;
			}
			temp[i] = 0;
			ltmp = atoi(temp) * LENGTH_SCALE;
			i = 0;
			while((a = ifile.get()) != '\n')
			{
				if(a == ' ') continue;
				if(a < '0' || a > '9')
					break;
				temp[i] = a;
				i++;
			}
			linecounter++;
			temp[i] = 0;
			if(ltmp && atoi(temp))
				ltmp /= atoi(temp);
			else	// M:C or M:C|
				ltmp = LENGTH_SCALE;
			if(verbose)
				std::cout << "Metrum: " << float(ltmp) / float(LENGTH_SCALE) << "\n";
			if(float(ltmp) / float(LENGTH_SCALE) < 0.75f)
				default_note = 1;
			else
				default_note = 2;
			if(verbose)
				std::cout << "Default note length set to: " << float(default_note) / float(LENGTH_SCALE) << " by M header entry.\n";
		}
		else if(a == 'L')	// Set default note length
		{
			if(ifile.get() != ':') continue;
			char temp[16] = "               ";
			i = 0;
			while((a = ifile.get()) != '/')
			{
				if(a == ' ') continue;
				temp[i] = a;
				i++;
			}
			temp[i] = 0;
			default_note = atoi(temp) * LENGTH_SCALE;
			i = 0;
			while((a = ifile.get()) != '\n')
			{
				if(a == ' ') continue;
				temp[i] = a;
				i++;
			}
			linecounter++;
			temp[i] = 0;		
			default_note /= atoi(temp);
			if(verbose)
				std::cout << "Default note length set to: " << float(default_note) / float(LENGTH_SCALE) << " by L header entry.\n";
std::cout.flush();
		}
		else if(a == 'Q')
		{
			line();
			std::cerr << "Warning: Setting speed not implemented yet (Q header entry).\n";
			while(ifile.get() != '\n');
			linecounter++;
		}
		else if(a == 'T')
		{
			ifile.get();	// Get ':'
			if(verbose)
				std::cout << "Current tune: ";
			while((a = ifile.get()) != '\n')
				if(verbose)
					std::cout << a;
			if(verbose)
				std::cout << '\n';
			linecounter++;
		}
		else if(a == 'C')
		{
			ifile.get();	// Get ':'
			if(verbose)
				std::cout << "Composer: ";
			while((a = ifile.get()) != '\n')
				if(verbose)
					std::cout << a;
			if(verbose)
				std::cout << '\n';
			linecounter++;
		}
		else if(a == 'S')
		{
			ifile.get();	// Get ':'
			if(verbose)
				std::cout << "Source: ";
			while((a = ifile.get()) != '\n')
				if(verbose)
					std::cout << a;
			if(verbose)
				std::cout << '\n';
			linecounter++;
		}
		else if(a == 'Z')
		{
			ifile.get();	// Get ':'
			if(verbose)
				std::cout << "Transcriber: ";
			while((a = ifile.get()) != '\n')
				if(verbose)
					std::cout << a;
			if(verbose)
				std::cout << '\n';
			linecounter++;
		}
		else if(a == 'N')
		{
			ifile.get();	// Get ':'
			if(verbose)
				std::cout << "Note: ";
			while((a = ifile.get()) != '\n')
				if(verbose)
					std::cout << a;
			if(verbose)
				std::cout << '\n';
			linecounter++;
		}
		else if(a == 'O')
		{
			ifile.get();	// Get ':'
			if(verbose)
				std::cout << "Origin: ";
			while((a = ifile.get()) != '\n')
				if(verbose)
					std::cout << a;
			if(verbose)
				std::cout << '\n';
			linecounter++;
		}
		else if(a == 'A')
		{
			ifile.get();	// Get ':'
			if(verbose)
				std::cout << "Author: ";
			while((a = ifile.get()) != '\n')
				if(verbose)
					std::cout << a;
			if(verbose)
				std::cout << '\n';
			linecounter++;
		}
		else if(a == '%')	// Ignore comments.
		{
			while(ifile.get() != '\n');
			linecounter++;
		}
		else if(a == 'X')	// Ignore X header when not using -X option.
		{
			while(ifile.get() != '\n');
			linecounter++;
		}
		else if(a == '\n')	// Ignore empty lines.
		{
			linecounter++;
		}
		else if (a == 'I')
		{
			std::string instruction, arg;
			ifile.get();	// Get ':'
			a = ifile.get();
			while(a != '\n' && a != ' ' && a != EOF)
			{
				instruction += a;
				a = ifile.get();
			}
			while(a == ' ')
				a = ifile.get();
			while(a != '\n' && a != ' ' && a != EOF)
			{
				arg += a;
				a = ifile.get();
			}
			if(instruction == "linebreak" && arg == "!")
				exclamationlinebreak = true;
			else if(instruction == "decoration" && arg == "+")
				{
					plusdecoration = true;
					exclamationlinebreak = true;
				}
			else if(verbose)
			{
				std::cout << "Warning: Ignoring unknown instruction: ";
				std::cout << instruction << " " << arg << "\n";
			}
			while(a != '\n' && a != EOF)
				ifile.get();
			linecounter++;
		}
		else
		{
			line();
			std::cerr << "Warning: Unknown header " << a << " ignored.\n";
			while(ifile.get() != '\n');
			linecounter++;
		}
	}

	if(!default_note)
	{
		std::cerr << "Error: Default note length not set.\n";
		return(-1);
	}

	ofile << "#include <stdint.h>\n" << "\n" << "const uint16_t notes[] = {";

	oldlinecounter = linecounter;
	repeatmark = ifile.tellg();
	repeatflag = false;
note.loudness = 2;	// Default to mf.
	erase();

	while((a = ifile.get()) != EOF)
	{
		switch(a)
		{
		case '^':
			translate();
			note.local_modifier = +1;
			note.local_modifier_enabled = true;
			break;
		case '_':
			translate();
			note.local_modifier = -1;
			note.local_modifier_enabled = true;
			break;
		case '=':
			translate();
			note.local_modifier = 0;
			note.local_modifier_enabled = true;
			break;
		case '.':
			translate();
			note.staccato = true;
			break;
		case '>':	// Broken rhythm
			if(note.dotted || note.halved)
			{
				line();
				std::cerr << "Warning: Unrecognized broken rhythm\n";
			}
			note.dotted = true;
			translate();
			note.halved = true;
			break;
		case '<':	// Broken rhythm
			if(note.dotted || note.halved)
			{
				line();
				std::cerr << "Warning: Unrecognized broken rhythm\n";
			}
			note.halved = true;
			translate();
			note.dotted = true;
			break;
		case 'z':
		case 'x':
		case 'a':
		case 'b':
		case 'c':
		case 'd':
		case 'e':
		case 'f':
		case 'g':
			translate();
			note.oktave = 3;
			note.base = toupper(a);
			break;
		case 'A':
		case 'B':
		case 'C':
		case 'D':
		case 'E':
		case 'F':
		case 'G':
			translate();
			note.oktave = 2;
			note.base = a;
			break;
		case ',':
			note.oktave--;
			break;
		case '\'':
			note.oktave++;
			break;
		case '-':
			note.local_legato = true;
			break;
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
			if(!note.base)
			{
				line();
				std::cerr << "Encountered number before pitch.\n";
			}
			else if(!note.denom_current)
				note.num[note.counter++] = a;
			else
				note.denom[note.counter++] = a;
			break;
		case '/':
			if(!note.denom_current)
			{
				note.denom_current = true;
				note.counter = 0;
				note.denom[note.counter] = 2;
			}
			else if(note.denom[note.counter] == 2)
			{
				note.counter = 0;
				note.denom[note.counter] = 4;
			}
			else if(note.denom[note.counter] == 4)
			{
				note.counter = 0;
				note.denom[note.counter] = 8;
			}
			else
			{
				line();
				std::cerr << "Warning: Failed to decode note length encoded by //...\n";
			}
			break;
		case '%':
			while(ifile.get() != '\n');	// Ignore comments.
			linecounter++;
			break;
		case '\"':
			while(ifile.get() != '\"');	// Skip over annotations.
			linecounter++;
			break;
		case '|':
			translate();
			for(int i = 0; i < 4; i++)
				for(int j = 0; j < 7; j++)
					local_modifier_enabled[i][j] = false;
			a = ifile.get();
			if(a == ':')
			{
				if(!skip)
				{
					oldlinecounter = linecounter;
					repeatmark = ifile.tellg();
					repeatflag = false;
				}
			}
			else if(a == '1')
			{
				if(repeatflag || skip)
				{
					char b;
					if(verbose && !skip)
					{
						line();
						std::cout << "Skipping variant 1 ending on repeat.\n";
					}
					do
					{
						b = a;
						a = ifile.get();
					}
					while(a != EOF && !((b == '|' || b == ']') && a == '2'));
					if(!skip)
					{
						oldlinecounter = linecounter;
						repeatmark = ifile.tellg();
						repeatflag = false;
					}
				}
			}
			else if(a != ']')
				ifile.unget();
			break;
		case '\n':
			linecounter++;
		case ' ':
		case '\t':
			break;	// Ignore whitespace.
		case '[':
			translate();

			// Handle beams.
			a = ifile.get();
			ifile.unget();
			if(a == '|')
				break;

			if(skip)
			{
				while(ifile.get() != ']');
				break;
			}
			switch(ifile.get())
			{
			case 'K':
				ifile.get();	// Skip :
				key(ifile, ']');
				break;
			default:
				line();
				std::cerr << "Warning: Encountered unimplemented inline [...] in music. Ignored.\n";
				while(ifile.get() != ']');
			}
			break;
		case 'K':	// Old style inline key
			ifile.get();	// Skip :
			if(skip)
			{
				while(ifile.get() != '\n');
				break;
			}
			key(ifile, '\n');
			break;
		case 'V':
			translate();
			ifile.get();	// Skip :
			if(voice == "")
			{
				line();
				std::cerr << "Warning: Ignoring voice specification, voice option not specified.\n";
			}
			else
			{
				std::string voice2;

				// Skip whitespace
				do
					a = ifile.get();
				while(a == ' ' || a == '\t');

				while(a != '\n' && a != ' ' && a != EOF)
				{
					voice2 += a;
					a = ifile.get();
				}

				if(voice != voice2)
				{
					if(verbose)
					{
						line();
						std::cout << "Skipping voice " << voice2 << ", looking for voice " << voice << "\n";
					}
					skip = true;
				}
				else
					skip = false;

				while(a != '\n' && a != EOF)
					a = ifile.get();
				if(a == '\n')
					linecounter++;
			}
			break;
		case '(':
			translate();
			a = ifile.get();
			if(a == '2')
			{
				line();
				std::cerr << "Warning: Encountered unimplemented duplet.\n";
			}
			else if(a == '3')
			{
				line();
				std::cerr << "Warning: Encountered unimplemented triplet.\n";
			}
			else if(a == '4')
			{
				line();
				std::cerr << "Warning: Encountered unimplemented quadruplet.\n";
			}
			else
			{
				ifile.unget();
				legato++;
			}
			break;
		case ')':
			legato--;
			if(legato < 0)
			{
				line();
				std::cerr << "Warning: Unmatched slur ending encountered.\n";
			}
			translate();
			break;
		case '{':
			translate();
			a = ifile.get();
			if(a == '/')
				note.acciaccatura = true;
			else
				ifile.unget();
			note.grace = true;
			break;
		case '}':
			translate();
			note.grace = false;
			break;
		case ':':
			translate();
			a = ifile.get();
			if(a != '|' && a != ':')
			{
				line();
				std::cerr << "Warning: Unrecognized repeat\n";
				oldlinecounter =linecounter;
				repeatmark = ifile.tellg();
				repeatflag = false;
				ifile.unget();
			}
			else
			{
				translate();
				a = ifile.get();
				if(a != '|' && a != ']')
					ifile.unget();
				if(skip)
					break;
				if(!repeatflag)
				{
					linecounter = oldlinecounter;
					repeatflag = true;
					ifile.seekg(repeatmark);
				}
				else
				{
					oldlinecounter = linecounter;
					repeatmark = ifile.tellg();
					repeatflag = false;
				}
			}
			break;
		case 'L':
			if(verbose)
				std::cout << "Encountered L, assuming accent.\n";
			note.accent = true;
			break;
		case '+': // Old syntax for decoration.
			{
				if(skip)
				{
					do
						a = ifile.get();
					while(a != '+' && a != EOF);
					break;
				}
				translate();
				char b, c, d = ' ';
				b = ifile.get();
				if(!plusdecoration)
				{
					line();
					std::cerr << "Warning: Deprecated decoration syntax.\n";
				}
				if(b != '+')
				{
					c = ifile.get();
					if(c != '+')
						d = ifile.get();
				}
				if(b != '+' && c != '+' && d != '+')
				{
					if(b == 't' && c == 'e' && d == 'n') // tenuto
						note.local_legato = true;
					else if(b == 't' && c == 'r' && d == 'i') // trill
					{
						line();
						std::cerr << "Warning: Ignoring unimplemented triller.\n";
					}
					else if(b == 't' && c == 'r' && d == 'i') // trill
					{
						line();
						std::cerr << "Warning: Ignoring unimplemented triller.\n";
					}
					do
						a = ifile.get();
					while(a != '!' && a != EOF);
					break;
				}
				if(b == '>' && c == '+')
					note.accent = true;
				else if(b == 'p' && c == '+')
					note.loudness = 0;
				else if(b == 'f' && c == '+')
					note.loudness = 3;
				else if(b == 'm' && c == 'p')
					note.loudness = 1;
				else if(b == 'm' && c == 'f')
					note.loudness = 2;
				else if((b == '<' || b == '>') && (c == '(' || c == ')'))
				{
					line();
					std::cerr << "Warning: Ignored unimplemented (de)crescendo.\n";
				}
				else
				{
					line();
					std::cerr << "Warning: Encountered unknown decoration.\n";
				}
			}
			break;
		case '!': // Old syntax for linebreaks, new syntax for decoration.
			if(exclamationlinebreak)
				break;
			{
				if(skip)
				{
					do
						a = ifile.get();
					while(a != '!' && a != EOF);
					break;
				}
				translate();
				char b, c, d = ' ';
				b = ifile.get();
				if(b != '!')
				{
					c = ifile.get();
					if(c != '!')
						d = ifile.get();
				}
				if(b != '!' && c != '!' && d != '!')
				{
					if(b == 't' && c == 'e' && d == 'n') // tenuto
						note.local_legato = true;
					else if(b == 't' && c == 'r' && d == 'i') // trill
					{
						line();
						std::cerr << "Warning: Ignoring unimplemented triller.\n";
					}
					else
					{
						line();
						std::cerr << "Warning: Ignoring unknown decoration.\n";
					}
					do
						a = ifile.get();
					while(a != '!' && a != EOF);
					break;
				}
				if(b == '>' && c == '!')
					note.accent = true;
				else if(b == 'p' && c == '!')
					note.loudness = 0;
				else if(b == 'f' && c == '!')
					note.loudness = 3;
				else if(b == 'm' && c == 'p')
					note.loudness = 1;
				else if(b == 'm' && c == 'f')
					note.loudness = 2;
				else if((b == '<' || b == '>') && (c == '(' || c == ')'))
				{
					line();
					std::cerr << "Warning: Ignored unimplemented (de)crescendo.\n";
				}
				else
				{
					line();
					std::cerr << "Warning: Encountered unknown decoration.\n";
				}
			}
			break;
		case '\\':
			break;
		case '$':
			break;
		case 'X':
			if(verbose)
			{
				line();
				std::cout << "End of tune.\n";
			}
			goto tune_end;
		default:
			line();
			std::cerr << "Warning: Encountered unimplemented " << a << " in music. Ignored.\n";
			break;
		}
	}
tune_end:
	translate();
	ofile << " 0xffff };\n";
	if(verbose)
		std::cout << "Wrote " << length << " notes, length in full notes: " << ((double)tlength / 16.0) << ".\n";
	ofile.close();
	ifile.close();
	return(0);
}

